/*-------------------------------------------------------------------------*
 *                                                                         *
 *   THIS IS AN UNPUBLISHED WORK CONTAINING CONFIDENTIAL AND PROPRIETARY   *
 *   INFORMATION.  IF PUBLICATION OCCURS, THE FOLLOWING NOTICE APPLIES:    *
 *        "COPYRIGHT 2012 AMARULA SOLUTIONS, ALL RIGHTS RESERVED"          *
 *                                                                         *
 *   NO DERIVATIVE WORKS ARE ALLOWED WITHOUT OUR EXPLICIT CONSENSOUS.      *
 *                                                                         *
 *   Amarula Solutions BV                                Cruquiuskade 47   *
 *   www.amarulasolutions.com                       Amsterdam 1018 AM NL   *
 *                                                                         *
 *-------------------------------------------------------------------------*/

package com.amarula.barcodescanner.sdk;

import android.content.ContentResolver;
import android.content.ContentValues;
import android.database.Cursor;
import android.net.Uri;
import android.provider.BaseColumns;
import android.util.Log;

public class BarcodeScannerSymbologyTable implements BaseColumns {

	private static final String TAG = "BarcodeScannerSymbologyTable";

	public static final String AUTHORITY = "com.amarula.barcodescanner.db";
	public static final String SCHEMA   = "content://";
	public static final String TABLE_NAME = "symbologies";

	public static final Uri CONTENT_URI = Uri.parse(SCHEMA + AUTHORITY + "/"+ TABLE_NAME);

	public static final String KEY_LABEL		= "label";			// text		Symbol Name Human readable
	public static final String KEY_HW_LABEL		= "hw_label";		// text		Hardware dependent label for this symbol
	public static final String KEY_ENABLED		= "enabled";		// integer
	public static final String KEY_CHANGED		= "changed";		// integer	If > 0 On next prepare() the scanner will sync options
	public static final String KEY_MIN_LENGTH	= "min_length";		// integer	Tag min length
	public static final String KEY_MAX_LENGTH	= "max_length";		// integer	Tag max length
	public static final String KEY_FLAGS		= "flags";			// integer	Hardware dependant flags

	private static String[] ALL_COLUMNS = {
		_ID, KEY_LABEL,
		KEY_HW_LABEL,
		KEY_ENABLED,
		KEY_CHANGED,
		KEY_MIN_LENGTH,
		KEY_MAX_LENGTH,
		KEY_FLAGS
	};

	private static final String DEFAULT_SORT_ORDER	= KEY_LABEL + " ASC";

	/**
	 * Return a Cursor that contains all symbols managed by the current
	 * hardware. <strong>Remember to close the cursor when no more needed</strong>
	 *
	 * @param cr: obtained by context.getContentResolver()
	 * @return: the resulting Cursor
	 */
	public static Cursor getAllSymbols(ContentResolver cr) {
		Cursor currentData = cr.query(CONTENT_URI, ALL_COLUMNS,
				null, null, DEFAULT_SORT_ORDER);
		if (currentData == null)
			Log.e(TAG, "Query returned null cursor!");

		return currentData;
	}

	/**
	 * Enable/Disable one symbol
	 *
	 * @param cr: obtained by context.getContentResolver()
	 * @param hwLabel: the label id that identify the symbol
	 * @param en: true to enable the symbol
	 */
	public static void setSymbolEnabled(ContentResolver cr,
			String hwLabel, boolean en) {
		ContentValues val = new ContentValues();

		val.put(KEY_ENABLED, en ? 1:0);
		val.put(KEY_CHANGED, 1);

		cr.update(CONTENT_URI, val,
				KEY_HW_LABEL + " = ?",
				new String[]{hwLabel});
		cr.notifyChange(CONTENT_URI, null);
	}

	/**
	 * Use this to disable all symbology and than enable one by one
	 * the symbols on need
	 *
	 * @param cr: obtained by context.getContentResolver()
	 */
	public static void disableAllSymbols(ContentResolver cr) {
		ContentValues val = new ContentValues();

		val.put(KEY_ENABLED, 0);
		val.put(KEY_CHANGED, 1);

		cr.update(CONTENT_URI, val,
				null, null);
		cr.notifyChange(CONTENT_URI, null);
	}

	/**
	 * Set Min/Max length properties of a symbol
	 *
	 * @param cr: obtained by context.getContentResolver()
	 * @param hwLabel: the label id that identify the symbol
	 * @param min: minimum tag length
	 * @param max: maximum tag length
	 */
	public static void setSymbolMinMaxLength(ContentResolver cr,
			String hwLabel, int min, int max) {
		ContentValues val = new ContentValues();

		val.put(KEY_MIN_LENGTH, min);
		val.put(KEY_MAX_LENGTH, max);
		val.put(KEY_CHANGED, 1);

		cr.update(CONTENT_URI, val,
				KEY_HW_LABEL + " = ?",
				new String[]{hwLabel});
		cr.notifyChange(CONTENT_URI, null);
	}

	/**
	 * Set symbol flags
	 *
	 * @param cr: obtained by context.getContentResolver()
	 * @param hwLabel: the label id that identify the symbol
	 * @param flags: hardware/symbol dependent flags
	 */
	public static void setSymbolFlags(ContentResolver cr,
			String hwLabel, int flags) {
		ContentValues val = new ContentValues();

		val.put(KEY_HW_LABEL, hwLabel);
		val.put(KEY_FLAGS, flags);
		val.put(KEY_CHANGED, 1);

		cr.update(CONTENT_URI, val,
				KEY_HW_LABEL + " = ?",
				new String[]{hwLabel});
		cr.notifyChange(CONTENT_URI, null);
	}

	public final class HwLabelHsm {
		public static final String HSM_AZTEC			= "z";
		public static final String HSM_CODABAR			= "a";
		public static final String HSM_CODE11			= "h";
		public static final String HSM_CODE128			= "j";
		public static final String HSM_CODE39			= "b";
		public static final String HSM_CODE49			= "l";
		public static final String HSM_CODE93			= "i";
		public static final String HSM_DATAMATRIX		= "w";
		public static final String HSM_EAN8				= "D";
		public static final String HSM_EAN13			= "d";
		public static final String HSM_EAN128			= "I";
		public static final String HSM_INTERLEAVED2OF5	= "e";
		public static final String HSM_MAXICODE			= "x";
		public static final String HSM_MICROPDF			= "R";
		public static final String HSM_PDF417			= "r";
		public static final String HSM_POSTNET			= "P";
		public static final String HSM_OCR				= "O";
		public static final String HSM_QR				= "s";
		public static final String HSM_GS1DATABAR		= "y";
		public static final String HSM_UPCA				= "c";
		public static final String HSM_UPCE				= "E";
		public static final String HSM_BRITISH_POST		= "B";
		public static final String HSM_CANADIAN_POST	= "C";
		public static final String HSM_AUSTRALIAN_POST	= "A";
		public static final String HSM_IATA25			= "f";
		public static final String HSM_CODABLOCK		= "q";
		public static final String HSM_JAPANESE_POST	= "J";
		public static final String HSM_PLANET_CODE		= "L";
		public static final String HSM_DUTCH_POST		= "K";
		public static final String HSM_MSI				= "g";
		public static final String HSM_TLC39			= "T";
		public static final String HSM_TRIOPTIC			= "=";
		public static final String HSM_CODE32			= "<";
		public static final String HSM_STRAIGHT2OF5		= "f";
		public static final String HSM_MATRIX2OF5		= "m";
		public static final String HSM_PLESSEY			= "n";
		public static final String HSM_CHINA_POST		= "Q";
		public static final String HSM_KOREA_POST		= "?";
		public static final String HSM_TELEPHEN			= "t";
		public static final String HSM_CODE16K			= "o";
		public static final String HSM_POSICODE			= "W";
		public static final String HSM_COUPON			= "c";
		public static final String HSM_USPS4CB			= "M";
		public static final String HSM_IDTAG			= "N";
		public static final String HSM_LABELIV			= ">";
		public static final String HSM_LABELV			= ",";
		public static final String HSM_GS1_128			= "I";
		public static final String HSM_HANXIN			= "H";
		public static final String HSM_GRIDMATRIX		= "x";
	}

	public final class FlagsHsm {
		// Flags to be used with Honeywell 2D barcode scanner
		public static final int HSM_CHECK_ENABLE		= 0x00000002; // Enable usage of check character
		public static final int HSM_CHECK_TRANSMIT		= 0x00000004; // Send check character
		public static final int HSM_START_STOP_XMIT		= 0x00000008; // Include the start and stop characters in the decoded result string
		public static final int HSM_ENABLE_APPEND_MODE	= 0x00000010; // Code39 append mode
		public static final int HSM_ENABLE_FULLASCII	= 0x00000020; // Enable Code39 Full ASCII
		public static final int HSM_NUM_SYS_TRANSMIT	= 0x00000040; // UPC-A/UPC-e Send Num Sys
		public static final int HSM_2_DIGIT_ADDENDA		= 0x00000080; // Enable 2 digit Addenda (UPC & EAN)
		public static final int HSM_5_DIGIT_ADDENDA		= 0x00000100; // Enable 5 digit Addenda (UPC & EAN)
		public static final int HSM_ADDENDA_REQUIRED	= 0x00000200; // Only allow codes with addenda (UPC & EAN)
		public static final int HSM_ADDENDA_SEPARATOR	= 0x00000400; // Include Addenda separator space in returned string.
		public static final int HSM_EXPANDED_UPCE		= 0x00000800; // Extended UPC-E
		public static final int HSM_UPCE1_ENABLE		= 0x00001000; // UPC-E1 enable (use HSM_ENABLE for UPC-E0)
		public static final int HSM_COMPOSITE_UPC		= 0x00002000; // Enable UPC composite codes
		public static final int HSM_AUSTRALIAN_BAR_WIDTH= 0x00010000; // Include australian postal bar data in string
		public static final int HSM_128_APPEND			= 0x00080000; // Enable OR dISBALE Code 128 FNC2 append functionality
		public static final int HSM_RSE_ENABLE			= 0x00800000; // Enable RSS Expanded bit
		public static final int HSM_RSL_ENABLE			= 0x01000000; // Enable RSS Limited bit
		public static final int HSM_RSS_ENABLE			= 0x02000000; // Enable RSS bit
		public static final int HSM_RSX_ENABLE_MASK		= 0x03800000; // Enable all RSS versions
		public static final int HSM_TELEPEN_OLD_STYLE	= 0x04000000; // Telepen Old Style mode.
		public static final int HSM_POSICODE_LIMITED_1	= 0x08000000; // PosiCode Limited of 1
		public static final int HSM_POSICODE_LIMITED_2	= 0x10000000; // PosiCode Limited of 2
		public static final int HSM_CODABAR_CONCATENATE	= 0x20000000; // Codabar concatenate.
	}

	public final class HwLabelN431x {
		public static final String N431X_CODABAR			= "a";
		public static final String N431X_CODE11				= "h";
		public static final String N431X_CODE128			= "j";
		public static final String N431X_CODE32				= "<";
		public static final String N431X_CODE39				= "b";
		public static final String N431X_CODE93				= "i";
		public static final String N431X_EAN13				= "d";
		public static final String N431X_EAN8				= "D";
		public static final String N431X_GS1DATABAR			= "y";
		public static final String N431X_GS1DATABAR_LIMITED	= "{";
		public static final String N431X_GS1DATABAR_EXPANDED = "}";
		public static final String N431X_GS1128				= "I";
		public static final String N431X_CHINAPOST			= "Q";
		public static final String N431X_INTERLEAVED2OF5	= "e";
		public static final String N431X_MATRIX1OF5			= "m";
		public static final String N431X_NEC2OF5			= "Y";
		public static final String N431X_STRAIGHT2OF5		= "f";
		public static final String N431X_MSI				= "g";
		public static final String N431X_TELEPHEN			= "t";
		public static final String N431X_UPCA				= "c";
		public static final String N431X_UPCE				= "E";
		public static final String N431X_AUSTRALIAN_POST	= "A";
		public static final String N431X_BRITISH_POST		= "B";
		public static final String N431X_CANADIAN_POST		= "C";
		public static final String N431X_CHINA_POST			= "Q";
		public static final String N431X_INFOMAIL			= ",";
		public static final String N431X_INTELLIGENTMAIL 	= "M";
		public static final String N431X_JAPANESE_POST		= "J";
		public static final String N431X_KIX				= "K";
		public static final String N431X_KOREA_POST			= "?";
		public static final String N431X_PLANET_CODE		= "L";
		public static final String N431X_POSTAL4I			= "N";
		public static final String N431X_POSTNET			= "P";
	}

	public final class FlagsN431x {
		public static final int N431X_CHECK_DISABLE				= 0x00000000; // To use with Codabar, UPC-A, UPC-E, EAN13, EAN8
		public static final int N431X_CHECK_ENABLE				= 0x00000001; // To use with UPC-A, UPC-E, EAN13, EAN8
		public static final int N431X_CHECK_ENABLE_TX			= 0x00000002; // To use with Code 39, Interleaved 2 of 5, NEC 2 of 5, Matrix 2 of 5
		public static final int N431X_CODABAR_VALID_MOD16		= 0x00000001; // Codabar Validate Modulo16, NOT Transmit
		public static final int N431X_CODABAR_VALID_MOD16_TX	= 0x00000002; // Codabar Validate Modulo16, Transmit
		public static final int N431X_CODABAR_VALID_MOD7		= 0x00000003; // Codabar Validate Modulo7 CD, NOT Transmit
		public static final int N431X_CODABAR_VALID_MOD7_TX		= 0x00000004; // Codabar Validate Modulo7 CD, Transmit
		public static final int N431X_CODABAR_VALID_CLSI		= 0x00000005; // Codabar Validate CLSI, NOT Transmit
		public static final int N431X_CODABAR_VALID_CLSI_TX		= 0x00000006; // Codabar Validate CLSI, Transmit
		public static final int N431X_MSI_VALID_TYPE10			= 0x00000001; // MSI Validate Type 10, NOT Transmit
		public static final int N431X_MSI_VALID_TYPE10_TX		= 0x00000002; // MSI Validate Type 10, Transmit
		public static final int N431X_MSI_VALID_2TYPE10			= 0x00000003; // MSI Validate 2 Type 10, NOT Transmit
		public static final int N431X_MSI_VALID_2TYPE10_TX		= 0x00000004; // MSI Validate 2 Type 10, Transmit
		public static final int N431X_CODE11_1CKDIGIT_REQ		= 0x00000000; // Code 11 One Check Digit Required
		public static final int N431X_CODE11_2CKDIGIT_REQ		= 0x00000001; // Code 11 Two Check Digits Required
		public static final int N431X_CODE11_CKDIGIT_AUTO		= 0x00000002; // Code 11 Auto Select Check Digits Required
	}

	public final class HwLabelSE965 {
		public static final String SE965_UPCA		= "0x08";
		public static final String SE965_UPCA_2		= "0x48";
		public static final String SE965_UPCA_5		= "0x88";
		public static final String SE965_UPCE		= "0x09";
		public static final String SE965_UPCE_2		= "0x49";
		public static final String SE965_UPCE_5		= "0x89";
		public static final String SE965_EAN8		= "0x0A";
		public static final String SE965_EAN8_2		= "0x4A";
		public static final String SE965_EAN8_5		= "0x8A";
		public static final String SE965_EAN13		= "0x0B";
		public static final String SE965_EAN13_2	= "0x4B";
		public static final String SE965_EAN13_5	= "0x8B";
		public static final String SE965_UPCE1		= "0x10";
		public static final String SE965_UPCE1_2	= "0x50";
		public static final String SE965_UPCE1_5	= "0x90";
		public static final String SE965_CCA		= "0x2F";
		public static final String SE965_CCB		= "0x37";
		public static final String SE965_BOOKLAND	= "0x16";
		public static final String SE965_COUPON		= "0x17";
		public static final String SE965_ISSN		= "0x36";
		public static final String SE965_CODE128	= "0x03";
		public static final String SE965_EAN128		= "0x0F";
		public static final String SE965_CODE39		= "0x01";
		public static final String SE965_TRIOPTIC	= "0x15";
		public static final String SE965_CODE39ASCII= "0x13";
		public static final String SE965_CODE32		= "0x20";
		public static final String SE965_CODE93		= "0x07";
		public static final String SE965_CODE11		= "0x0C";
		public static final String SE965_INT25		= "0x05";
		public static final String SE965_ITF		= "0x06";
		public static final String SE965_D25		= "0x04";
		public static final String SE965_MATRIX25	= "0x39";
		public static final String SE965_KOREAN25	= "0x73";
		public static final String SE965_CODABAR	= "0x02";
		public static final String SE965_MSI		= "0x0E";
		public static final String SE965_GS1DATABAR	= "0x32";
		public static final String SE965_ISBT128	= "0x19";
		public static final String SE965_ISBT128CON	= "0x21";
		// Unknown ids starting from here
		public static final String SE965_CHINESE25	= "0xFE";
		public static final String SE965_UPCEAN_SUPPL = "0xFF"; // Fake id to use to set UPC/EAN supplementals settings
	}

	public final class OffsetsSE965 {
		public static final int SE965_CHK_DIG				= 0; // 2b 0: EN 1: DIS
																 // OR MSI:   0: MSI One MSI Check Digit 1: MSI Two MSI Check Digit
																 // OR INT25: 0: Disable 1: USS 2: OPCC
																 // OR COD11: 0: Disable 1: One 2: Two
		public static final int SE965_CHK_DIG_TX			= 2; // 1b 0: No Trasmit 1: Transmit
		public static final int SE965_MSI_CHK_ALGO			= 3; // 1b 0: MOD11/MOD10 1: MOD10/MOD10
		public static final int SE965_ISBT128_CHK_TABLE		= 4; // 1b 0: Disabe 1: Enable
		public static final int SE965_CONCAT_SUPP			= 5; // 4b 0: Disabe 1: Enable 2: Autodiscriminate
																 // OR UPC/EAN Suppl: 0: Ignore 1: Decode Only with Supplementals 2: Autodiscriminate
																 //   3: Enable Smart Mode 4: Enable 378/379 Mode 5: Enable 978/979 Mode
																 //   6: Enable 414/419/434/439 Mode 7: Enable 977 Mode 8: Enable 491 Mode
		public static final int SE965_CONCAT_SUPP_REDUNDANCY= 9; // 5b Redundancy min: 2 max: [ISBT 20] [UPC/EAN 30]
		public static final int SE965_ISBN_FORMAT			=14; // 1b Bookland ISBN format: 0: ISBN-10 1: ISBN-13
	}


	public final class HwLabelSDL {
		public static final String SDL_CODE_39				= "0x01";
		public static final String SDL_CODABAR				= "0x02";
		public static final String SDL_CODE_128				= "0x03";
		public static final String SDL_DISCRETE_2_OF_5		= "0x04";
		public static final String SDL_INTERLEAVED_2_OF_5	= "0x06";
		public static final String SDL_CODE_93				= "0x07";
		public static final String SDL_UPC_A				= "0x08";
		public static final String SDL_UPC_E				= "0x09";
		public static final String SDL_EAN_8				= "0x0A";
		public static final String SDL_EAN_13				= "0x0B";
		public static final String SDL_CODE_11				= "0x0C";
		public static final String SDL_MSI					= "0x0E";
		public static final String SDL_EAN_128				= "0x0F";
		public static final String SDL_UPC_E1				= "0x10";
		public static final String SDL_PDF_417				= "0x11";
		public static final String SDL_CODE39_FULL_ASCII	= "0x13";
		public static final String SDL_CODE39_TRIOPTIC		= "0x15";
		public static final String SDL_BOOKLAND_EAN 		= "0x16";
		public static final String SDL_ISBT_128				= "0x19";
		public static final String SDL_MICRO_PDF			= "0x1A";
		public static final String SDL_DATA_MATRIX			= "0x1B";
		public static final String SDL_QR_CODE				= "0x1C";
		public static final String SDL_US_POSTNET			= "0x1E";
		public static final String SDL_US_PLANET			= "0x1F";
		public static final String SDL_CODE_32				= "0x1F";
		public static final String SDL_JAPAN_POSTAL			= "0x22";
		public static final String SDL_AUSTRALIA_POSTAL		= "0x23";
		public static final String SDL_DUTCH_POSTAL			= "0x24";
		public static final String SDL_MAXICODE 			= "0x25";
		public static final String SDL_UK_POSTAL			= "0x27";
		public static final String SDL_MICRO_QR				= "0x2C";
		public static final String SDL_AZTEC				= "0x2D";
		public static final String SDL_GS1_DATABAR_14		= "0x30";
		public static final String SDL_GS1_DATABAR_LIMITED	= "0x31";
		public static final String SDL_GS1_DATABAR_EXPANDED	= "0x32";
		public static final String SDL_USPS_4CB				= "0x34";
		public static final String SDL_UPU_4STATE			= "0x35";
		public static final String SDL_ISSN_EAN				= "0x36";
		public static final String SDL_MATRIX_2_OF_5		= "0x39";
		public static final String SDL_CHINESE_2_OF_5		= "0x72";
		public static final String SDL_KOREAN_3_OF_5		= "0x73";
		public static final String SDL_HAN_XIN				= "0xB7";
	}

	public final class FlagsSDL {
		public static final int SDL_FLAG_NONE					 = -1;
		public static final int SDL_CHK_DIG_TX_OFFSET			= 16;
		public static final int SDL_CHK_DIG_TX_DIS				= 0;
		public static final int SDL_CHK_DIG_TX_EN				= 1;
		public static final int SDL_CHK_DIG_TX_INVALID			= 0xf;
		public static final int SDL_SET_CHK_DIG_EN				= (SDL_CHK_DIG_TX_EN << SDL_CHK_DIG_TX_OFFSET);
		public static final int SDL_SET_CHK_DIG_DIS				= (SDL_CHK_DIG_TX_DIS << SDL_CHK_DIG_TX_OFFSET);
		public static final int SDL_SET_CHK_DIG_INVALID			= (SDL_CHK_DIG_TX_INVALID << SDL_CHK_DIG_TX_OFFSET);

		public static final int SDL_CHK_DGT_OFFSET				= 0;
		public static final int SDL_CODE39_CHK_DGT_VERI_DIS		= 0; // Default
		public static final int SDL_CODE39_CHK_DGT_VERI_EN		= 1;

		public static final int SDL_CODE39_FULL_ASCII_OFFSET	= 4;
		public static final int SDL_CODE39_FULL_ASCII_DIS		= 0; // Default
		public static final int SDL_CODE39_FULL_ASCII_EN		= 1;

		public static final int SDL_CODE39_CONVERT_32_OFFSET	= 8;
		public static final int SDL_CODE39_CONVERT_32_DIS		= 0; // Default
		public static final int SDL_CODE39_CONVERT_32_EN		= 1;
		public static final int SDL_CODE39_DEFAULT_FLAG			= (SDL_SET_CHK_DIG_DIS |
																	(SDL_CODE39_CHK_DGT_VERI_DIS << SDL_CHK_DGT_OFFSET) |
																	(SDL_CODE39_FULL_ASCII_DIS << SDL_CODE39_FULL_ASCII_OFFSET) |
																	(SDL_CODE39_CONVERT_32_DIS << SDL_CODE39_CONVERT_32_OFFSET));

		public static final int SDL_MSI_CHK_DGT_TYPE_ONE		= 0; // Default
		public static final int SDL_MSI_CHK_DGT_TYPE_TWO		= 1;

		public static final int SDL_MSI_CHK_DGT_ALGO_OFFSET		= 4;
		public static final int SDL_MSI_CHK_DGT_ALGO_MODE10_10	= 1; // Default
		public static final int SDL_MSI_CHK_DGT_ALGO_MODE10_11	= 0;

		public static final int SDL_MSI_DEFAULT_FLAG			= (SDL_SET_CHK_DIG_DIS |
																	(SDL_MSI_CHK_DGT_TYPE_ONE << SDL_CHK_DGT_OFFSET) |
																	(SDL_MSI_CHK_DGT_ALGO_MODE10_10 << SDL_MSI_CHK_DGT_ALGO_OFFSET) );

		public static final int SDL_CODE11_CHK_DGT_VERY_DIS		= 0; // Deafault
		public static final int SDL_CODE11_CHK_DGT_VERY_1CD		= 1;
		public static final int SDL_CODE11_CHK_DGT_VERY_2CD		= 2;
		public static final int SDL_CODE11_DEFAULT_FLAG			= (SDL_SET_CHK_DIG_DIS |
																	(SDL_CODE11_CHK_DGT_VERY_DIS << SDL_CHK_DGT_OFFSET));

		public static final int SDL_I25_CHK_DIG_DISABLE			= 0; // Default
		public static final int SDL_I25_CHK_DIG_USS				= 1;
		public static final int SDL_I25_CHK_DIG_OPCC			= 2;

		public static final int SDL_I25_CONVERT_OFFSET			= 4;
		public static final int SDL_I25_CONVERT_EAN13_EN		= 1;
		public static final int SDL_I25_CONVERT_EAN13_DIS		= 0; // Default

		public static final int SDL_I25_SECURITY_OFFSET			= 8;
		public static final int SDL_I25_SECURITY_LEVEL0			= 0;
		public static final int SDL_I25_SECURITY_LEVEL1			= 1; // Default
		public static final int SDL_I25_SECURITY_LEVEL2			= 2;
		public static final int SDL_I25_SECURITY_LEVEL3			= 3;
		public static final int SDL_I25_DEFAULT_FLAG			= (SDL_SET_CHK_DIG_DIS |
																	(SDL_I25_CHK_DIG_DISABLE << SDL_CHK_DGT_OFFSET) |
																	(SDL_I25_CONVERT_EAN13_DIS << SDL_I25_CONVERT_OFFSET) |
																	(SDL_I25_SECURITY_LEVEL1 << SDL_I25_SECURITY_OFFSET));

		public static final int SDL_ISBT_CONCATE_OFFSET			= 0;
		public static final int SDL_ISBT_CONCATE_DIS			= 0; // Default
		public static final int SDL_ISBT_CONCATE_EN				= 1;
		public static final int SDL_ISBT_CONCATE_AUTODISC		= 2;

		public static final int SDL_ISBT_TABLE_OFFSET			= 4;
		public static final int SDL_ISBT_TABLE_DIS				= 0;
		public static final int SDL_ISBT_TABLE_EN				= 1; // Default
		public static final int SDL_ISBT_DEFAULT_FLAG			= (SDL_SET_CHK_DIG_INVALID |
																	(SDL_ISBT_CONCATE_DIS << SDL_ISBT_CONCATE_OFFSET) |
																	(SDL_ISBT_TABLE_EN << SDL_ISBT_TABLE_OFFSET));


		public static final int SDL_MATRIX_25_CHK_DGT_DIS		= 0; // Default
		public static final int SDL_MATRIX_25_CHK_DGT_EN		= 1;

		public static final int SDL_MATRIX_25_REDUNDANCY_OFFSET	= 4;
		public static final int SDL_MATRIX_25_REDUNDANCY_DIS	= 0; // Default
		public static final int SDL_MATRIX_25_REDUNDANCY_EN		= 1;
		public static final int SDL_MATRIX_25_DEFAULT_FLAG		= (SDL_SET_CHK_DIG_DIS |
																	(SDL_MATRIX_25_CHK_DGT_DIS << SDL_CHK_DGT_OFFSET) |
																	(SDL_MATRIX_25_REDUNDANCY_DIS << SDL_MATRIX_25_REDUNDANCY_OFFSET));

		public static final int SDL_AUS_POST_FMT_OFFSET			= 0;
		public static final int SDL_AUS_POST_FMT_AUTODISC		= 0; // Default
		public static final int SDL_AUS_POST_FMT_ROW			= 1;
		public static final int SDL_AUS_POST_FMT_ALPHANUMERIC	= 2;
		public static final int SDL_AUS_POST_FMT_NUMERIC		= 3;
		public static final int SDL_AUS_POST_DEFAULT_FLAG		= (SDL_SET_CHK_DIG_INVALID |
																	(SDL_AUS_POST_FMT_AUTODISC << SDL_AUS_POST_FMT_OFFSET));


		public static final int SDL_INVERSE_OFFSET				= 0;
		public static final int SDL_INVERSE_REGULAR				= 0; // Default
		public static final int SDL_INVERSE_INVERSE				= 1;
		public static final int SDL_INVERSE_AUTO				= 2;

		public static final int SDL_DATA_MATRIX_MIRROR_OFFSET	= 4;
		public static final int SDL_DATA_MATRIX_MIRROR_NEVER 	= 0; // Default
		public static final int SDL_DATA_MATRIX_MIRROR_ALWAYS	= 1;
		public static final int SDL_DATA_MATRIX_MIRROR_AUTO		= 2;
		public static final int SDL_DATA_MATRIX_DEFAULT_FLAG	= (SDL_SET_CHK_DIG_INVALID |
																	(SDL_INVERSE_REGULAR << SDL_INVERSE_OFFSET) |
																	(SDL_DATA_MATRIX_MIRROR_NEVER << SDL_DATA_MATRIX_MIRROR_OFFSET));

		public static final int SDL_QR_DEFAULT_FLAG				= (SDL_SET_CHK_DIG_INVALID |
																	(SDL_INVERSE_REGULAR << SDL_INVERSE_OFFSET));

		public static final int SDL_AZTEC_DEFAULT_FLAG			= (SDL_SET_CHK_DIG_INVALID |
																	(SDL_INVERSE_REGULAR << SDL_INVERSE_OFFSET));

		public static final int SDL_HAN_XIN_DEFAULT_FLAG		= (SDL_SET_CHK_DIG_INVALID |
																	(SDL_INVERSE_REGULAR << SDL_INVERSE_OFFSET));

		public static final int SDL_BOOKLAND_ISBN_FORMAT_OFFSET	= 0;
		public static final int SDL_BOOKLAND_ISBN_FORMAT_10		= 0; // Deafult
		public static final int SDL_BOOKLAND_ISBN_FORMAT_13		= 1;
		public static final int SDL_BOOKLAND_ISBN_DEFAULT_FLAG	= (SDL_SET_CHK_DIG_INVALID |
																	(SDL_BOOKLAND_ISBN_FORMAT_10 << SDL_BOOKLAND_ISBN_FORMAT_OFFSET));

		public static final int SDL_CODE32_PREFIX_OFFSET		= 0;
		public static final int SDL_CODE32_PREFIX_DIS			= 0; // Deafult
		public static final int SDL_CODE32_PREFIX_EN			= 1;
		public static final int SDL_CODE32_DEFAULT_FLAG			= (SDL_SET_CHK_DIG_INVALID |
																	(SDL_CODE32_PREFIX_DIS << SDL_CODE32_PREFIX_OFFSET));
	}
}

